
// BlobTorus.h
// Definit la classe des metaballs (isosurface d'un tore)

#ifndef V3D_BLOBS_BLOB_TORUS_H_INCLUDED
#define V3D_BLOBS_BLOB_TORUS_H_INCLUDED

#include "BlobTransformable.h"


namespace V3D {

//////////////////////////////////////////////////////////////////////////////////////////
class BlobTorus : public BlobTransformable
{
	typedef BlobTransformable parent;

	BLOBTRANSFORMABLE_STD_OVERRIDES( BlobTorus );

public:
	// Construction / Destruction
	BlobTorus( float fCenterEnergy, const Vector3D& vcPos, float fInfluenceRange, float fLargeRadius = 1.f, float fOpenAngle = 0.f);
	virtual ~BlobTorus();

	virtual bool GetBlobBounds( Vector3D& vcBlobMin, Vector3D& vcBlobMax ) const;

	virtual inline bool  EarlyRejectBaseShapePoint( const Vector3D& vcPos ) const;
	virtual inline float GetBaseShapeSqrDist( const Vector3D& pos ) const;
	virtual inline void  ConvertBaseShapePosToGradient( Vector3D& vcOffs ) const;
	

	virtual void EnergyBaseShapeAddAtRectangle ( float* pafAddEnergies,    // Tableau des valeurs energetiques de la grille
										 const Vector3D& vcRectStart,  // Origine du rectangle
										 const Vector3D& vcRectAxisX,  // Axe X du rectangle (sert d'unite de separation entre deux calculs d'energie)
										 const Vector3D& vcRectAxisY,  // Axe Y du rectangle (sert d'unite de separation entre deux calculs d'energie)
                                         int32 nLinesCount,                    // Nombre de subdivisions en Y dans le rectangle
										 int32 nLineLength,                    // Nombre de subdivisions en X dans le rectangle
										 int32 nSizeX                          // Nombre de subdivisions en X dans la grille
										) const;  


protected:
	float		m_fLargeRadius;
	float		m_fOpenAngle;



};

//////////////////////////////////////////////////////////////////////////////////////////


inline float BlobTorus::GetBaseShapeSqrDist(const Vector3D& pos)const
{ 
/*
	if( m_fOpenAngle < k_fPi )
	{
		float fAngle = Atan2(pos.y, pos.x);
		if( Abs(fAngle) > m_fOpenAngle )
		{
			float fDiffEdgeX = m_fLargeRadius * Cos(m_fOpenAngle);
			float fDiffEdgeY = m_fLargeRadius * Sin(m_fOpenAngle);
			float fResult = (fDiffEdgeX - pos.x)*(fDiffEdgeX - pos.x) + pos.z * pos.z;
//			if( fAngle > 0 )
				return fResult + (fDiffEdgeY - pos.y)*(fDiffEdgeY - pos.y);
//			else
//				return fResult + (fDiffEdgeY - pos.y)*(fDiffEdgeY - pos.y);
		}
	}
	else
*/
	{
		LengthType rTmp = Sqrt(pos.x * pos.x + pos.y * pos.y) - m_fLargeRadius;
		return float( rTmp * rTmp + pos.z * pos.z );
	}
}
						/////////////////////////////

inline void  BlobTorus::ConvertBaseShapePosToGradient( Vector3D& vcOffs) const
{
	LengthType rRadCircle = Sqrt(vcOffs.x * vcOffs.x+ vcOffs.y * vcOffs.y);
//	float fTmp = fRadCircle - m_fLargeRadius;
	LengthType rFactXY = 1.f - m_fLargeRadius / rRadCircle;
	vcOffs.x *= rFactXY;
	vcOffs.y *= rFactXY;
}

						/////////////////////////////

//inline bool BlobTorus::EarlyRejectPoint(const Vector3D& pos) const
//{
//	return false;
//}

						/////////////////////////////

inline bool BlobTorus::EarlyRejectBaseShapePoint(const Vector3D&  pos ) const
{
	return (    ( Abs( pos.z) > 1.f)
	         || ( ( pos.x * pos.x + pos.y * pos.y) > ((m_fLargeRadius + 1.f) * (m_fLargeRadius + 1.f)) ) 
	         || ( ( pos.x * pos.x + pos.y * pos.y) < ((m_fLargeRadius - 1.f) * (m_fLargeRadius - 1.f)) ) );
}

						/////////////////////////////




/*

// Retourne l'energie generee par la boule au point 'pos'
inline bool BlobTorus::EnergyAddAtPos( float& fDest, const Vector3D& pos) const
{
	// On evite les appels aux methodes, pour garantir de bonnes performances
	float fDiffX = pos.x - m_vcPosition.x;
	float fDiffY = pos.y - m_vcPosition.y;
	float fDiffZ = pos.z - m_vcPosition.z;
	float fTmp = Sqrt(fDiffX * fDiffX + fDiffY * fDiffY) - m_fLargeRadius;
	float fRadInflSqr = fTmp * fTmp + fDiffZ * fDiffZ;

	if( m_fOpenAngle < k_fPi )
	{
		float fAngle = Atan2(fDiffY, fDiffX);
		if( Abs(fAngle) > m_fOpenAngle )
		{
			float fDiffEdgeX = m_fLargeRadius * Cos(m_fOpenAngle);
			float fDiffEdgeY = m_fLargeRadius * Sin(m_fOpenAngle);
			fRadInflSqr = (fDiffEdgeX - fDiffX)*(fDiffEdgeX - fDiffX) + fDiffZ * fDiffZ;
			if( fAngle > 0 )
				fRadInflSqr += (fDiffEdgeY - fDiffY)*(fDiffEdgeY - fDiffY);
			else
				fRadInflSqr += (fDiffEdgeY - fDiffY)*(fDiffEdgeY - fDiffY);
		}
	}
//	Premier test: si le point est hors du tore d'influence, inutile de continuer
	if( fRadInflSqr < m_fInfluenceRangeSqr )
		fDest += EnergyNotZeroGetAtSqrDist( fRadInflSqr);
}



inline void BlobTorus::GradientAddAtOnePos( Vector3D& vcGrad, const Vector3D& pos) const
{

	// On evite les appels de fonction comme vcOffs.GetSquareLength()
	// Car le compilateur ne garantit pas de la mettre en inline
	float fDiffX = pos.x - m_vcPosition.x;
	float fDiffY = pos.y - m_vcPosition.y;
	float fDiffZ = pos.z - m_vcPosition.z;
	float fRadCircle = Sqrt(fDiffX * fDiffX + fDiffY * fDiffY);
	float fTmp = fRadCircle - m_fLargeRadius;
	float fSqrDist = fTmp * fTmp + fDiffZ * fDiffZ;
	float fSqrDistDivInfl = fSqrDist * m_fInfluenceRangeInvSqr;


	// Rayon >= rayon de seuil ==> energie = 0 
	if( fSqrDistDivInfl >= 1.f)
		return;

	float fFactXY = 1.f - m_fLargeRadius / fRadCircle;
	Vector3D vcOffs( fDiffX * fFactXY, fDiffY * fFactXY, fDiffZ);

	float fFactor;

	if( fSqrDistDivInfl > (1.f / 9.f) )
	{
		// Rayon entre rayon de seuil / 3 et rayon de seuil ==> energie = formule 2
		float fFactorTmp = 3.f * m_fFullEnergy * m_fInfluenceRangeInv;
		fFactor = fFactorTmp * (( 1.f / Sqrt(fSqrDist) ) - m_fInfluenceRangeInv );
	}
	else
	{
		// Rayon < rayon de seuil / 3 ==> energie = formule 1
		fFactor = 6.f * m_fFullEnergy * m_fInfluenceRangeInvSqr;
	}
	// Idem pour les performances de l'operateur *= de la classe Vector3D
	vcOffs.x *= fFactor;
	vcOffs.y *= fFactor;
	vcOffs.z *= fFactor;
	vcGrad += vcOffs;
}
*/

//////////////////////////////////////////////////////////////////////////////////////////

} // namespace




#endif	// #ifndef V3D_BLOBS_BLOB_TORUS_H_INCLUDED
